<?php

namespace App\Http\Controllers\Admin;

use Illuminate\Http\Request;
use Illuminate\Support\Facades\DB;
use App\Models\MonthlyConfig;
use App\Models\GeneralConfig;
use App\Http\Controllers\Controller;
use Yajra\DataTables\DataTables;
use Carbon\Carbon;
use App\Models\InvestHistory;
use App\Models\Referral;
use App\Models\User;

use DateTime;
class MonthlyConfigController extends Controller
{
    public function index()
    {
        return view('admin.monthly_config.index');
    }

    public function list(Request $request)
    {
        $monthlyPlans = MonthlyConfig::query();

        return DataTables::of($monthlyPlans)
        ->editColumn('period_start', function($plan) {
            return $plan->period_start ? $plan->period_start->format('Y-m-d') : '-';
        })        
        ->editColumn('period_end', function($plan) {
            return $plan->period_end ? $plan->period_end->format('Y-m-d') : '-';
        })
        ->editColumn('cost', function($plan) {
            return $plan->cost ? number_format($plan->cost,  4, '.', '') : '0.0000';
        })
        ->editColumn('reward', function($plan) {
            return $plan->reward ? number_format($plan->reward,  4, '.', '') : '0.0000';
        })
        ->editColumn('total_machine', function($plan) {
            return $plan->total_machine;
        })
        ->editColumn('operation_percentage', function($plan) {
            return $plan->operation_percentage;
        })
        ->editColumn('created_at', function($plan) {
            return $plan->created_at->format('Y-m-d H:i:s');
        })   
        ->editColumn('updated_at', function($plan) {
            return $plan->updated_at->format('Y-m-d H:i:s');
        })       
        ->editColumn('status', function($plan) {
            return $plan->status == 1
            ? '<span class="badge bg-success">Confirmed</span>'
            : '<span class="badge bg-danger text-warning">Waiting Confirmation</span>';
        })
        ->rawColumns(['status']) 

        ->addColumn('action', function($plan) {
            $editBtn = '<a href="'.route('admin.monthly.plan.edit', $plan->id).'" class="btn btn-sm btn-outline-primary" title="Edit">
            <i class="bi bi-pencil-square"></i>
            </a>';

            $viewBtn = '<a href="'.route('admin.monthly.plan.overview', $plan->id).'" class="btn btn-sm btn-outline-info" title="View">
            <i class="bi bi-eye"></i>
            </a>';
            $reportDropdown = '
            <div class="btn-group">
            <button type="button" class="btn btn-sm btn-outline-info dropdown-toggle" data-bs-toggle="dropdown" title="Reports">
            <i class="bi bi-bar-chart-line"></i>
            </button>
            <ul class="dropdown-menu">
            <li>
            <a class="dropdown-item" href="'.route('admin.monthly.plan.referral.commission', $plan->id).'">Referral Commission Report</a>
            </li>
            <li>
            <a class="dropdown-item" href="'.route('admin.monthly.plan.referral.summary', $plan->id).'">Mining Reward Report</a>
            </li>       
            <li>
            <a class="dropdown-item" href="'.route('admin.monthly.plan.summary_report', $plan->id).'">Summary</a>
            </li>
            </ul>
            </div>';
            
            $viewBtns = '<a href="'.route('admin.monthly.plan.summary_report', $plan->id).'" class="btn btn-sm btn-outline-info" title="Summary">
            <i class="bi bi-cash"></i>
            </a>';

            $deleteForm = '<form action="'.route('admin.monthly.plan.destroy', $plan->id).'" method="POST" class="d-inline">
            '.csrf_field().'
            '.method_field('DELETE').'
            <button type="submit" class="btn btn-sm btn-outline-danger" onclick="return confirm(\'Are you sure you want to delete this plan?\')" title="Delete">
            <i class="bi bi-trash"></i>
            </button>
            </form>';
            if ($plan->status == 1) {
                $deleteForm ='';
            }
            return '<div class="btn-group" role="group">'
            .$editBtn
            .$viewBtn
            .$reportDropdown
            .$viewBtns
            .$deleteForm.
            '</div>';
        })

        ->rawColumns(['status', 'action'])
        ->make(true);
    }

    public function create()
    {
        return view('admin.monthly_config.create');
    }

    public function store(Request $request)
    {
        $request->validate([
            'period_start' => 'required|date',
            'period_end' => 'required|date|after:period_start',
            'reward' => 'required|numeric|min:0|decimal:0,4',
            'total_machine' => 'required|numeric|min:0',
            'operation_percentage' => 'required|numeric|min:0',
            'cost' => 'required|numeric|min:0|decimal:0,4',
        ]);

        MonthlyConfig::create([
            'period_start' => $request->period_start,
            'period_end' => $request->period_end,
            'reward' => $request->reward,
            'total_machine' => $request->total_machine,
            'operation_percentage' => $request->operation_percentage,
            'cost' => $request->cost,
        ]);

        return redirect()->route('admin.monthly.plan.index')->with('success', 'Monthly config created successfully.');
    }

    public function edit($id)
    {
        $config = MonthlyConfig::findOrFail($id);
        return view('admin.monthly_config.edit', compact('config'));    
    }  

    private function getHashrateForPeriod($start, $end)
    {
        $general = GeneralConfig::where('period_start', '<=', $start)
        ->where('period_end', '>=', $end)
        ->first();

        return $general ? $general->hashrate : 0;
    }

    private function getTotalMachinesDuringPeriod($period_end)
    {
        return DB::table('invest_histories')
        ->whereDate('created_at', '<=', $period_end)
        ->whereDate('plan_expiry_date', '>=', $period_end)
        ->sum('quantity');
    }

    private function calculateRewards($config, $start, $end)
    {
        $total_days = $start->diffInDays($end) + 1;
        $reward = (float) $config->reward;
        $cost = (float) $config->cost;
        $operation_percentage = $config->operation_percentage;
        $total_machines = $config->total_machine;
        $gross_reward=$reward - $cost;
        $after_gross = $gross_reward - ($gross_reward * ($operation_percentage / 100));

        $full_reward=$gross_reward / ($total_machines * $total_days);
        $sub_reward=$after_gross / ($total_machines * $total_days);

        return [
            'reward_per_day'     => $full_reward,
            'sub_reward_per_day'     => $sub_reward,
        ];
    }

    public function summary($id)
    {
        $config = MonthlyConfig::findOrFail($id);
        $status = $config->status;
        $configId = $id;

        $period_start = Carbon::parse($config->period_start);
        $period_end = Carbon::parse($config->period_end);

        $hashrate = $this->getHashrateForPeriod($period_start, $period_end);

        $result = $this->calculateRewards($config, $period_start, $period_end);
        $reward_per_day=$result['reward_per_day'];
        $sub_reward_per_day=$result['sub_reward_per_day'];

        $investments = DB::table('invest_histories')
        ->select([
            'invest_histories.id',
            'invest_histories.user_id',
            'invest_histories.plan_id',
            'invest_histories.invest_amount',
            'invest_histories.quantity',
            'invest_histories.created_at',
            'invest_histories.trx',
            DB::raw("CONCAT(users.firstname, ' ', users.lastname) AS user_name"),
            'investment_plans.plan_name'
        ])
        ->join('users', 'users.id', '=', 'invest_histories.user_id')
        ->join('investment_plans', 'investment_plans.id', '=', 'invest_histories.plan_id')
        ->whereDate('invest_histories.created_at', '<=', $period_end)
        ->whereDate('invest_histories.plan_expiry_date', '>=', $period_end)
        ->orderBy('invest_histories.created_at', 'desc')
        ->get();

        $total_days_invested = 0;
        $total_reward = 0;
        $total_quantity=0;
        $processedInvestments = $investments->map(function ($investment) use ($period_start, $period_end, $sub_reward_per_day,$hashrate,&$total_days_invested, &$total_reward,&$total_quantity) {
            $investmentDate = Carbon::parse($investment->created_at); 
            $purchased_date = $investmentDate->format('Y-m-d');

            $today = Carbon::today();
            $reward_cutoff = $today->lessThan($period_end) ? $today : $period_end;

            $proposed_start = $investmentDate->copy()->addDay();
            $effective_start = $proposed_start->greaterThan($period_start) ? $proposed_start : $period_start;

            $effective_end = $reward_cutoff;

            $effective_start = $effective_start->copy()->startOfDay();
            $effective_end = $effective_end->copy()->startOfDay();

            $days_invested = $effective_start->lte($effective_end)
            ? $effective_start->diffInDays($effective_end) + 1
            : 0;

            $calculated_reward = $sub_reward_per_day * $days_invested * $investment->quantity;


        // Update totals
            $total_days_invested += $days_invested;
            $total_reward += $calculated_reward;
            $total_quantity += $investment->quantity;

            return [
                'id' => $investment->id,
                'user_id' => $investment->user_id,
                'planId' => $investment->plan_id,
                'user_name' => $investment->user_name,
                'invest_amount' => $investment->invest_amount,
                'quantity' => $investment->quantity,
                'days_invested' => $days_invested,
                'calculated_reward' => number_format($calculated_reward, 4, '.', ''),
                'purchased_date' => $purchased_date,
                'effective_start_date' => $effective_start->format('Y-m-d'), 
                'effective_end_date' => $period_end->format('Y-m-d'), 
                'hashrate' => $hashrate, 
                'trx' => $investment->trx, 
            ];
        });
        $total_days = $period_start->diffInDays($period_end) + 1;
        $summary = [
            'start_date' => $period_start->format('Y-m-d'),
            'end_date' => $period_end->format('Y-m-d'),
            'total_date_period' => $total_days,
            'total_reward_setting' => $config->reward,
            'total_operation_cost' =>$config->cost,
            'reward_per_day' => number_format($sub_reward_per_day, 4, '.', ''),
            'total_investments' => count($processedInvestments),
            'total_days_invested' => $total_days_invested,
            'total_quantity' => $total_quantity,
            'total_reward' => number_format($total_reward, 4, '.', ''),
        ];
        $cacheKey = 'processedInvestments_' . $id;
        \Cache::forever($cacheKey, $processedInvestments);
        return view('admin.monthly_config.summary', compact('processedInvestments', 'summary','status','configId'));
    }
    public function summaryPerUser($id,$user_id)
    {
        $config = MonthlyConfig::findOrFail($id);
        $status = $config->status;
        $configId = $id;

        $period_start = Carbon::parse($config->period_start);
        $period_end = Carbon::parse($config->period_end);

        $hashrate = $this->getHashrateForPeriod($period_start, $period_end);

        $result = $this->calculateRewards($config, $period_start, $period_end);
        $reward_per_day=$result['reward_per_day'];
        $sub_reward_per_day=$result['sub_reward_per_day'];

        $investments = DB::table('invest_histories')
        ->select([
            'invest_histories.id',
            'invest_histories.user_id',
            'invest_histories.plan_id',
            'invest_histories.invest_amount',
            'invest_histories.quantity',
            'invest_histories.created_at',
            DB::raw("CONCAT(users.firstname, ' ', users.lastname) AS user_name"),
            'investment_plans.plan_name'
        ])
        ->join('users', 'users.id', '=', 'invest_histories.user_id')
        ->join('investment_plans', 'investment_plans.id', '=', 'invest_histories.plan_id')
        ->whereDate('invest_histories.created_at', '<=', $period_end)
        ->whereDate('invest_histories.plan_expiry_date', '>=', $period_end)
        ->where('invest_histories.user_id', $user_id)
        ->orderBy('invest_histories.created_at', 'desc')
        ->get();

        $total_days_invested = 0;
        $total_reward = 0;
        $total_quantity=0;
        $processedInvestments = $investments->map(function ($investment) use ($period_start, $period_end, $sub_reward_per_day,$hashrate,&$total_days_invested, &$total_reward,&$total_quantity) {
            $investmentDate = Carbon::parse($investment->created_at); 
            $purchased_date = $investmentDate->format('Y-m-d');

            $today = Carbon::today();
            $reward_cutoff = $today->lessThan($period_end) ? $today : $period_end;

            $proposed_start = $investmentDate->copy()->addDay();
            $effective_start = $proposed_start->greaterThan($period_start) ? $proposed_start : $period_start;

            $effective_end = $reward_cutoff;

            $effective_start = $effective_start->copy()->startOfDay();
            $effective_end = $effective_end->copy()->startOfDay();

            $days_invested = $effective_start->lte($effective_end)
            ? $effective_start->diffInDays($effective_end) + 1
            : 0;

            $calculated_reward = $sub_reward_per_day * $days_invested * $investment->quantity;

        // Update totals
            $total_days_invested += $days_invested;
            $total_reward += $calculated_reward;
            $total_quantity += $investment->quantity;

            return [
                'id' => $investment->id,
                'user_id' => $investment->user_id,
                'planId' => $investment->plan_id,
                'user_name' => $investment->user_name,
                'invest_amount' => $investment->invest_amount,
                'quantity' => $investment->quantity,
                'days_invested' => $days_invested,
                'calculated_reward' => number_format($calculated_reward, 4, '.', ''),
                'purchased_date' => $purchased_date,
                'effective_start_date' => $effective_start->format('Y-m-d'), 
                'effective_end_date' => $period_end->format('Y-m-d'), 
                'hashrate' => $hashrate, 
            ];
        });
        $total_days = $period_start->diffInDays($period_end) + 1;
        $summary = [
            'start_date' => $period_start->format('Y-m-d'),
            'end_date' => $period_end->format('Y-m-d'),
            'total_date_period' => $total_days,
            'total_reward_setting' => $config->reward,
            'total_operation_cost' =>$config->cost,
            'reward_per_day' => number_format($sub_reward_per_day,  4, '.', ''),
            'total_investments' => count($processedInvestments),
            'total_days_invested' => $total_days_invested,
            'total_quantity' => $total_quantity,
            'total_reward' => number_format($total_reward,  4, '.', ''),
        ];
        return view('admin.monthly_config.summaryuser', compact('processedInvestments', 'summary','status','configId'));
    }
    private function calculateTotalCommission($userReferrals) {
        $total = 0;

        if (!isset($userReferrals['level1'])) {
            return $total;
        }

        foreach ($userReferrals['level1'] as $level1) {
            $total += $level1['commission'];

            if (!empty($level1['level2'])) {
                foreach ($level1['level2'] as $level2) {
                    $total += $level2['commission'];

                    if (!empty($level2['level3'])) {
                        foreach ($level2['level3'] as $level3) {
                            $total += $level3['commission'];
                        }
                    }
                }
            }
        }

        return $total;
    }

    public function updateSummary($id) {
        header('Content-Type: application/json; charset=utf-8');

        $referralReport = \Cache::get('referralReport_' . $id);
        $commissionMap = [];

        foreach ($referralReport as $userId => $userData) {
            $commissionMap[$userId] = $this->calculateTotalCommission($userData);
        }

        $mining_report = \Cache::get('processedInvestments_' . $id);
        $grouped = [];

        foreach ($mining_report as $record) {
            $userId = $record['user_id'];

            if (!isset($grouped[$userId])) {
                $grouped[$userId] = $record;
            } else {
                $grouped[$userId]['calculated_reward'] += $record['calculated_reward'];
                $grouped[$userId]['quantity'] += $record['quantity'];
                $grouped[$userId]['days_invested'] += $record['days_invested'];
            }
        }

        foreach ($grouped as $userId => &$rec) {
            $reward = $rec['calculated_reward'];
            $commission = isset($commissionMap[$userId]) ? $commissionMap[$userId] : 0;

            $rec['calculated_reward'] = number_format($reward, 4, '.', '');
            $rec['total_commission'] = number_format($commission, 4, '.', '');
            $rec['total_reward'] = number_format($reward + $commission, 4, '.', '');

            unset($rec['id'], $rec['planId'], $rec['invest_amount'], $rec['effective_start_date'], $rec['effective_end_date']);

        // Insert or update into DB
            $data = [
                'parent_id' => $id,
                'user_id' => $userId,
                'name' => $rec['user_name'],
                'mining_reward' => $rec['calculated_reward'],
                'hashrate' => $rec['hashrate'],
                'commission_reward' => $rec['total_commission'],
                'total_reward' => $rec['total_reward'],
                'total_machines' => $rec['quantity'],
                'days_invested' => $rec['days_invested'],
                'updated_at' => date('Y-m-d H:i:s')
            ];

            \DB::table('summary')->updateOrInsert(
                [
                    'user_id' => $userId,
                    'parent_id' => $id
                ],
                $data 
            );
        }

        echo json_encode([
            'success' => true,
            'message' => 'Summary updated and stored successfully.',
            'data' => array_values($grouped)
        ]);

        exit;
    }



    public function overview($id)
    {
        $config = MonthlyConfig::findOrFail($id);
        $status = $config->status;
        $configId = $id;

        $period_start = Carbon::parse($config->period_start);
        $period_end = Carbon::parse($config->period_end);

        $hashrate = $this->getHashrateForPeriod($period_start, $period_end);

        $result = $this->calculateRewards($config, $period_start, $period_end);

        $reward_per_day=$result['reward_per_day'];
        $sub_reward_per_day=$result['sub_reward_per_day'];

        $investments = DB::table('invest_histories')
        ->select([
            'invest_histories.id',
            'invest_histories.user_id',
            'invest_histories.plan_id',
            'invest_histories.invest_amount',
            'invest_histories.quantity',
            'invest_histories.created_at',
            DB::raw("CONCAT(users.firstname, ' ', users.lastname) AS user_name"),
            'investment_plans.plan_name'
        ])
        ->join('users', 'users.id', '=', 'invest_histories.user_id')
        ->join('investment_plans', 'investment_plans.id', '=', 'invest_histories.plan_id')
        ->whereDate('invest_histories.created_at', '<=', $period_end)
        ->whereDate('invest_histories.plan_expiry_date', '>=', $period_end)
        ->orderBy('invest_histories.created_at', 'desc')
        ->get();

        $total_days_invested = 0;
        $total_reward = 0;
        $total_quantity=0;

        $processedInvestments = $investments->map(function ($investment) use ($period_start, $period_end, $sub_reward_per_day, $hashrate, &$total_days_invested, &$total_reward, &$total_quantity) {

            $investmentDate = Carbon::parse($investment->created_at); 
            $purchased_date = $investmentDate->format('Y-m-d');

            $today = Carbon::today();
            $reward_cutoff = $today->lessThan($period_end) ? $today : $period_end;

            $proposed_start = $investmentDate->copy()->addDay();
            $effective_start = $proposed_start->greaterThan($period_start) ? $proposed_start : $period_start;

            $effective_end = $reward_cutoff;

            $effective_start = $effective_start->copy()->startOfDay();
            $effective_end = $effective_end->copy()->startOfDay();

            $days_invested = $effective_start->lte($effective_end)
            ? $effective_start->diffInDays($effective_end) + 1
            : 0;

            $calculated_reward = $sub_reward_per_day * $days_invested * $investment->quantity;

            $total_days_invested += $days_invested;
            $total_reward += $calculated_reward;
            $total_quantity += $investment->quantity;

            return [
                'id' => $investment->id,
                'user_id' => $investment->user_id,
                'planId' => $investment->plan_id,
                'user_name' => $investment->user_name,
                'invest_amount' => $investment->invest_amount,
                'quantity' => $investment->quantity,
                'days_invested' => $days_invested,
                'calculated_reward' => number_format($calculated_reward, 4, '.', ''),
                'purchased_date' => $purchased_date,
                'effective_start_date' => $effective_start->format('Y-m-d'),
                'effective_end_date' => $effective_end->format('Y-m-d'),
                'hashrate' => $hashrate,
            ];
        });


        $total_days = $period_start->diffInDays($period_end) + 1;
        $summary = [
            'start_date' => $period_start->format('Y-m-d'),
            'end_date' => $period_end->format('Y-m-d'),
            'total_date_period' => $total_days,
            'total_reward_setting' => $config->reward,
            'total_operation_cost' =>$config->cost,
            'reward_per_day' => number_format($sub_reward_per_day,  4, '.', ''),
            'total_investments' => count($processedInvestments),
            'total_days_invested' => $total_days_invested,
            'total_quantity' => $total_quantity,
            'total_reward' => number_format($total_reward,  4, '.', ''),
        ];

        return view('admin.monthly_config.calculator', compact('processedInvestments', 'summary','status','configId'));
    }

    public function referralCommission($id)
    {
        $config = MonthlyConfig::findOrFail($id);
        $status = $config->status;
        $configId = $id;

        $period_start = Carbon::parse($config->period_start);
        $period_end = Carbon::parse($config->period_end);

        $hashrate = $this->getHashrateForPeriod($period_start, $period_end);
        $total_machines = $this->getTotalMachinesDuringPeriod($period_end);

        $result = $this->calculateRewards($config, $period_start, $period_end);
        $reward_per_day=$result['reward_per_day'];
        $sub_reward_per_day=$result['sub_reward_per_day'];

    // Get the investments within the given period
        $investments = DB::table('users')
        ->select([
            'invest_histories.id',
            'users.id as user_id',
            'invest_histories.plan_id',
            'invest_histories.invest_amount',
            'invest_histories.quantity',
            'invest_histories.created_at',
            'users.email',
            DB::raw("CONCAT(users.firstname, ' ', users.lastname) AS user_name"),
            'investment_plans.plan_name'
        ])
        ->leftJoin('invest_histories', function($join) use ($period_end) {
            $join->on('users.id', '=', 'invest_histories.user_id')
            ->whereDate('invest_histories.created_at', '<=', $period_end)
            ->whereDate('invest_histories.plan_expiry_date', '>=', $period_end);
        })
        ->leftJoin('investment_plans', 'investment_plans.id', '=', 'invest_histories.plan_id')
        ->orderBy('users.id')
        ->get();


        //this is the calculation of the each user transaction
        $processedInvestments = $investments->map(function ($investment) use ($period_start, $period_end, $reward_per_day, $hashrate) {
            if ($investment->created_at) {
                $investmentDate = Carbon::parse($investment->created_at); 
                $purchased_date = $investmentDate->format('Y-m-d');

                $today = Carbon::today();
                $reward_cutoff = $today->lessThan($period_end) ? $today : $period_end;

                $proposed_start = $investmentDate->copy()->addDay();
                $effective_start = $proposed_start->greaterThan($period_start) ? $proposed_start : $period_start;

                $effective_end = $reward_cutoff;

                $effective_start = $effective_start->copy()->startOfDay();
                $effective_end = $effective_end->copy()->startOfDay();

                $days_invested = $effective_start->lte($effective_end)
                ? $effective_start->diffInDays($effective_end) + 1
                : 0;

                $calculated_reward = $reward_per_day * $days_invested * $investment->quantity;
            } else {
                $days_invested = 0;
                $purchased_date = '';
                $calculated_reward = '';
                $effective_start = '';
            }

            return [
                'id' => $investment->id ?? '',
                'user_id' => $investment->user_id,
                'planId' => $investment->plan_id ?? '',
                'user_name' => $investment->user_name,
                'invest_amount' => $investment->invest_amount ?? '',
                'quantity' => $investment->quantity ?? '',
                'days_invested' => $days_invested,
                'calculated_reward' => $calculated_reward !== '' ? number_format($calculated_reward, 4, '.', '') : '',
                'purchased_date' => $purchased_date,
                'effective_start_date' => $effective_start !== '' ? $effective_start->format('Y-m-d') : '',
                'effective_end_date' => $investment->created_at ? $period_end->format('Y-m-d') : '',
                'hashrate' => $investment->created_at ? $hashrate : '',
            ];
        });


    // Pre-define the level 1~3 percentage
        $referralCommissions = Referral::where('commission_type', 'profit_commission')
        ->whereIn('level', [1, 2, 3])
        ->pluck('commission', 'level');

        //get only the agent approved user
        $users=User::pluck('id');
        //loop the users
        $referralReport=array();
        foreach($users as $user){
            $checklvl = getDirectReferralUsers($user);
            $countofref=$checklvl->count();
            if($countofref == 0){
                continue;
            }
            $userData = User::find($user);
            $referralReport[$user]['agent'] = [
                'user_id' => $userData->id,
                'user_name' => $userData->firstname.' '.$userData->lastname,
                'user_email' => $userData->email,
                'purchased_date' => null,
                'start_date' => null,
                'end_date' => null,
                'quantity' => null,
                'commission' => null,
                'referral_name' => null,
            ];

            $result = $this->processLevel1Referrals($user, $processedInvestments, $referralCommissions);
            $referralReport[$user]['level1'] = $result['level1'];
        }
        $cacheKey = 'referralReport_' . $id;
        \Cache::forever($cacheKey, $referralReport);
        return view('admin.monthly_config.commission', compact(
            'referralReport',
        ));
    }
    public function referralCommissionPerUser($id,$user_id)
    {
        $config = MonthlyConfig::findOrFail($id);
        $status = $config->status;
        $configId = $id;

        $period_start = Carbon::parse($config->period_start);
        $period_end = Carbon::parse($config->period_end);

        $hashrate = $this->getHashrateForPeriod($period_start, $period_end);
        $total_machines = $this->getTotalMachinesDuringPeriod($period_end);

        $result = $this->calculateRewards($config, $period_start, $period_end);
        $reward_per_day=$result['reward_per_day'];
        $sub_reward_per_day=$result['sub_reward_per_day'];

    // Get the investments within the given period
        $investments = DB::table('users')
        ->select([
            'invest_histories.id',
            'users.id as user_id',
            'invest_histories.plan_id',
            'invest_histories.invest_amount',
            'invest_histories.quantity',
            'invest_histories.created_at',
            'users.email',
            DB::raw("CONCAT(users.firstname, ' ', users.lastname) AS user_name"),
            'investment_plans.plan_name'
        ])
        ->leftJoin('invest_histories', function($join) use ($period_end) {
            $join->on('users.id', '=', 'invest_histories.user_id')
            ->whereDate('invest_histories.created_at', '<=', $period_end)
            ->whereDate('invest_histories.plan_expiry_date', '>=', $period_end);
        })
        ->leftJoin('investment_plans', 'investment_plans.id', '=', 'invest_histories.plan_id')
        ->orderBy('users.id')
        ->get();

        //this is the calculation of the each user transaction
        $processedInvestments = $investments->map(function ($investment) use ($period_start, $period_end, $reward_per_day, $hashrate) {
            if ($investment->created_at) {
                $investmentDate = Carbon::parse($investment->created_at); 
                $purchased_date = $investmentDate->format('Y-m-d');

                $today = Carbon::today();
                $reward_cutoff = $today->lessThan($period_end) ? $today : $period_end;

                $proposed_start = $investmentDate->copy()->addDay();
                $effective_start = $proposed_start->greaterThan($period_start) ? $proposed_start : $period_start;

                $effective_end = $reward_cutoff;

                $effective_start = $effective_start->copy()->startOfDay();
                $effective_end = $effective_end->copy()->startOfDay();

                $days_invested = $effective_start->lte($effective_end)
                ? $effective_start->diffInDays($effective_end) + 1
                : 0;

                $calculated_reward = $reward_per_day * $days_invested * $investment->quantity;
            } else {
                $days_invested = 0;
                $purchased_date = '';
                $calculated_reward = '';
                $effective_start = '';
            }

            return [
                'id' => $investment->id ?? '',
                'user_id' => $investment->user_id,
                'planId' => $investment->plan_id ?? '',
                'user_name' => $investment->user_name,
                'invest_amount' => $investment->invest_amount ?? '',
                'quantity' => $investment->quantity ?? '',
                'days_invested' => $days_invested,
                'calculated_reward' => $calculated_reward !== '' ? number_format($calculated_reward, 4, '.', '') : '',
                'purchased_date' => $purchased_date,
                'effective_start_date' => $effective_start !== '' ? $effective_start->format('Y-m-d') : '',
                'effective_end_date' => $investment->created_at ? $period_end->format('Y-m-d') : '',
                'hashrate' => $investment->created_at ? $hashrate : '',
            ];
        });

        $referralCommissions = Referral::where('commission_type', 'profit_commission')
        ->whereIn('level', [1, 2, 3])
        ->pluck('commission', 'level');

        $users=User::where('id',$user_id)->pluck('id');
        
        $referralReport=array();
        foreach($users as $user){
            $checklvl = getDirectReferralUsers($user);
            $countofref=$checklvl->count();
            if($countofref == 0){
                continue;
            }
            $userData = User::find($user);
            $referralReport[$user]['agent'] = [
                'user_id' => $userData->id,
                'user_name' => $userData->firstname.' '.$userData->lastname,
                'user_email' => $userData->email,
                'purchased_date' => null,
                'start_date' => null,
                'end_date' => null,
                'quantity' => null,
                'commission' => null,
                'referral_name' => null,
            ];

            $result = $this->processLevel1Referrals($user, $processedInvestments, $referralCommissions);
            $referralReport[$user]['level1'] = $result['level1'];
        }
        return view('admin.monthly_config.commissionuser', compact(
            'referralReport',
        ));
    }
    private function processLevel1Referrals($userId, $userInvestments, $referralCommissions)
    {
        $level1Users = getDirectReferralUsers($userId);
        $level1Data = [];

        foreach ($level1Users as $level1User) {
            $rewardEntry = $userInvestments->firstWhere('user_id', $level1User->id);
            if (!$rewardEntry) continue;
            $reward = $rewardEntry ? (float)$rewardEntry['calculated_reward'] : 0;
            $commission = ($referralCommissions[1] ?? 0) / 100 * $reward;

            $level2Result = $this->processLevel2Referrals($level1User->id, $userInvestments, $referralCommissions);

            $level1Data[] = [
                'user_id' => $level1User->id,
                'user_email' => $level1User->email,
                'user_name' => $level1User->firstname . ' ' . $level1User->lastname,
                'reward' => $reward,
                'commission_rate' => $referralCommissions[1] ?? 0,
                'commission' => $commission,
                'purchased_date' => $rewardEntry['purchased_date'],
                'start_date' => $rewardEntry['effective_start_date'],
                'end_date' => $rewardEntry['effective_end_date'],
                'quantity' => $rewardEntry['quantity'],
                'level2' => $level2Result['level2']
            ];

        }

        return [
            'level1' => $level1Data,
        ];
    }

    private function processLevel2Referrals($level1UserId, $userInvestments, $referralCommissions)
    {
        $level2Users = getDirectReferralUsers($level1UserId);
        $level2Data = [];

        foreach ($level2Users as $level2User) {
            $rewardEntry = $userInvestments->firstWhere('user_id', $level2User->id);
            if (!$rewardEntry) continue;
            $reward = $rewardEntry ? (float)$rewardEntry['calculated_reward'] : 0;
            $commission = ($referralCommissions[2] ?? 0) / 100 * $reward;

            $level3Result = $this->processLevel3Referrals($level2User->id, $userInvestments, $referralCommissions);

            $level2Data[] = [
                'user_id' => $level2User->id,
                'user_email' => $level2User->email,
                'user_name' => $level2User->firstname . ' ' . $level2User->lastname,
                'reward' => $reward,
                'commission_rate' => $referralCommissions[2] ?? 0,
                'commission' => $commission,
                'purchased_date' => $rewardEntry['purchased_date'],
                'start_date' => $rewardEntry['effective_start_date'],
                'end_date' => $rewardEntry['effective_end_date'],
                'quantity' => $rewardEntry['quantity'],
                'level3' => $level3Result['level3']
            ];
        }

        return [
            'level2' => $level2Data,
        ];
    }

    private function processLevel3Referrals($level2UserId, $userInvestments, $referralCommissions)
    {
        $level3Users = getDirectReferralUsers($level2UserId);
        $level3Data = [];

        foreach ($level3Users as $level3User) {
            $rewardEntry = $userInvestments->firstWhere('user_id', $level3User->id);
            if (!$rewardEntry) continue;
            $reward = $rewardEntry ? (float)$rewardEntry['calculated_reward'] : 0;
            $commission = ($referralCommissions[3] ?? 0) / 100 * $reward;

            $level3Data[] = [
                'user_id' => $level3User->id,
                'user_email' => $level3User->email,
                'user_name' => $level3User->firstname . ' ' . $level3User->lastname,
                'reward' => $reward,
                'commission_rate' => $referralCommissions[3] ?? 0,
                'commission' => $commission,
                'purchased_date' => $rewardEntry['purchased_date'],
                'start_date' => $rewardEntry['effective_start_date'],
                'end_date' => $rewardEntry['effective_end_date'],
                'quantity' => $rewardEntry['quantity']
            ];

        }

        return [
            'level3' => $level3Data,
        ];
    }



    public function update(Request $request, $id)
    {
        $request->validate([
            'period_start' => 'required|date',
            'period_end' => 'required|date|after:period_start',
            'reward' => 'required|numeric|min:0|decimal:0,4',
            'total_machine' => 'required|numeric|min:0',
            'operation_percentage' => 'required|numeric|min:0',
            'cost' => 'required|numeric|min:0|decimal:0,4',
        ]);

        $config = MonthlyConfig::findOrFail($id);
        $config->update([
            'period_start' => $request->period_start,
            'period_end' => $request->period_end,
            'reward' => $request->reward,
            'total_machine' => $request->total_machine,
            'operation_percentage' => $request->operation_percentage,
            'cost' => $request->cost,
        ]);

        return redirect()->route('admin.monthly.plan.index')->with('success', 'Monthly config updated successfully.');
    }

    public function destroy($id)
    {
        $config = MonthlyConfig::findOrFail($id);
        $config->delete();

        return redirect()->route('admin.monthly.plan.index')->with('success', 'Monthly config deleted successfully.');
    }

    public function approve($id)
    {
        $plan = MonthlyConfig::findOrFail($id);
        $plan->status = 1;
        $plan->save();

        return redirect()->route('admin.monthly.plan.index')->with('success', 'Configuration approved successfully.');
    }

}