<?php

namespace App\Http\Controllers\Admin;

use Illuminate\Http\Request;
use Illuminate\Support\Facades\DB;
use App\Models\MonthlyConfig;
use App\Models\GeneralConfig;
use App\Http\Controllers\Controller;
use Yajra\DataTables\DataTables;
use Carbon\Carbon;
use App\Models\InvestHistory;
use App\Models\Referral;
use App\Models\User;
use Illuminate\Support\Facades\Log;
use App\Traits\Notify;

class MonthlyReportController extends Controller
{    
    use Notify;


    public function summary_report($id)
    {
        return view('admin.monthly_report.summary');

    }

    public function listsummary(Request $request)
    {
        $monthlyPlans = \DB::table('summary')
        ->where('parent_Id', $request->id)
        ->get();

        return DataTables::of($monthlyPlans)
        ->editColumn('status', function ($plan) {
            return $plan->status == 1
            ? '<span class="badge bg-success">Confirmed</span>'
            : '<span class="badge bg-danger text-warning">Waiting Confirmation</span>';
        })
        ->addColumn('action', function ($plan) {
            $viewMiningReport = route('admin.monthly.plan.referral.summaryuser', [
                'user_id' => $plan->user_id,
                'id' => $plan->parent_id
            ]);

            $viewCommissionReport = route('admin.monthly.plan.referral.commissionuser', [
                'user_id' => $plan->user_id,
                'id' => $plan->parent_id
            ]);

            $transferRoute = route('admin.monthly.plan.transfer', [
                'user_id' => $plan->user_id,
                'id' => $plan->parent_id
            ]);

            $reportDropdown = '
            <div class="btn-group me-1">
            <button type="button" class="btn btn-sm btn-outline-info dropdown-toggle" data-bs-toggle="dropdown" title="Reports">
            <i class="bi bi-bar-chart-line"></i>
            </button>
            <ul class="dropdown-menu">
            <li><a class="dropdown-item" href="' . $viewMiningReport . '">Mining Reward Report</a></li>
            <li><a class="dropdown-item" href="' . $viewCommissionReport . '">Referral Commission Report</a></li>
            </ul>
            </div>';

   // Check if wallet address exists
            $wallet = \DB::table('user_address')->where('user_id', $plan->user_id)->first();

            // Transfer button only if wallet exists
            $transferBtn = '';
            if ($wallet && $wallet->address) {
                $transferBtn = '
                <a href="' . $transferRoute . '" class="btn btn-sm btn-outline-primary" title="Transfer">
                    <i class="bi bi-arrow-left-right"></i>
                </a>';
            }

            return '
            <div class="btn-group" role="group">
            ' . $reportDropdown . '
            ' . $transferBtn . '
            </div>';
        })

        ->rawColumns(['status', 'action'])
        ->make(true);
    }

    public function resendCode(Request $request)
    {
        $userId = session('verify_user_id');

        if (!$userId) {
            return redirect()->route('admin.login')->with('error', 'Session expired. Please login again.');
        }

        $user = \App\Models\Admin::find($userId);

        if (!$user) {
            return redirect()->route('admin.login')->with('error', 'User not found.');
        }

        $user->payment_code = rand(100000, 999999);
        $user->save();

        $this->verifyToMail($user, 'TWO_FACTOR_AUTH_PAYMENT', [
            'code' => $user->payment_code
        ]);

        return back()->with('status', 'A new verification code has been sent to your email.');
    }

    public function showVerifyForm()
    {
        return view('admin.monthly_report.verify'); 
    }

    public function verifyCode(Request $request)
    {
        $request->validate([
            'verify_code' => 'required|digits:6'
        ]);

        $userId = session('verify_user_id');
        $user = \App\Models\Admin::find($userId);

        if (!$user || $user->payment_code !== $request->verify_code) {
            return back()->withErrors(['verify_code' => 'Invalid verification code.']);
        }

        $user->verified = true;
        $user->payment_code = null;
        $user->save();

        session()->forget('verify_user_id');
        $this->bulkApprove($user->summary_ids);
        return redirect(route('admin.monthly.plan.index'))->with('status', 'Approved transaction.');

    }

    public function approveTrans(Request $request)
    {
        $log = Log::channel('doge_tx');

        if (empty($request->strIds)) {
            session()->flash('error', 'You did not select any users.');
            $log->warning("No user selected for bulk approval.");
            return response()->json(['error' => 1]);
        }

        $user = \App\Models\Admin::find(auth()->id());
        session(['verify_user_id' => auth()->id()]);

        if (!$user) {
            return response()->json(['error' => 1, 'message' => 'Admin user not found']);
        }

        $user->payment_code = rand(100000, 999999);
        $user->summary_ids = json_encode($request->strIds);
        $user->save();

        $this->verifyToMail($user, 'TWO_FACTOR_AUTH_PAYMENT', [
            'code' => $user->payment_code
        ]);

        return response()->json([
            'success' => true,
            'redirect_url' => route('admin.monthly.verify.form'),
            'message' => 'Verification code sent to email.'
        ]);
    }


    public function transfer($id,$user_id)
    {
        $referralReport=$this->referralCommissionPerUser($id,$user_id);
        $processedInvestments=$this->summaryPerUser($id,$user_id);
        $monthlyPlansUser = \DB::table('summary')
        ->where('parent_Id', $id)
        ->where('user_id', $user_id)
        ->first();
        $wallet = \DB::table('user_address')->where('user_id', $user_id)->first();
        $hash = \DB::table('withdraw_log')->where('summary_id', $monthlyPlansUser->id)->first();

        return view('admin.monthly_report.transfer', compact(
            'referralReport',
            'processedInvestments',
            'monthlyPlansUser',
            'wallet',
            'id',
            'user_id',
            'hash',
        ));
    }

    public function maketransfer($id,$user_id,Request $request)
    {
        $monthlyPlansUser = \DB::table('summary')
        ->where('parent_Id', $id)
        ->where('user_id', $user_id)
        ->first();
        $wallet = \DB::table('user_address')->where('user_id', $user_id)->first();
        $trx=$request->transaction_hash;
        DB::table('withdraw_log')->insert([
            'parent_id'     => $monthlyPlansUser->parent_id,
            'user_id'       => $monthlyPlansUser->user_id,
            'tx_id'         => $trx,
            'to_address'    => $wallet->address,
            'amount'        => $monthlyPlansUser->total_reward,
            'from_address'  => '',
            'status'        => 1,
            'fee'           => 0,  
            'summary_id'           => $monthlyPlansUser->id,  
            'created_at'    => now(),
            'updated_at'    => now(),
        ]);

        DB::table('summary')->where('id', $monthlyPlansUser->id)->update(['status' => 1]);
        return redirect()->route('admin.monthly.plan.summary_report', $id)->with('success', 'Approved Transaction.');
    }
    private function getHashrateForPeriod($start, $end)
    {
        $general = GeneralConfig::where('period_start', '<=', $start)
        ->where('period_end', '>=', $end)
        ->first();

        return $general ? $general->hashrate : 0;
    }

    private function getTotalMachinesDuringPeriod($period_end)
    {
        return DB::table('invest_histories')
        ->whereDate('created_at', '<=', $period_end)
        ->whereDate('plan_expiry_date', '>=', $period_end)
        ->sum('quantity');
    }

    private function calculateRewards($config, $start, $end)
    {
        $total_days = $start->diffInDays($end) + 1;
        $reward = (float) $config->reward;
        $cost = (float) $config->cost;
        $operation_percentage = $config->operation_percentage;
        $total_machines = $config->total_machine;
        $gross_reward=$reward - $cost;
        $after_gross = $gross_reward - ($gross_reward * ($operation_percentage / 100));

        $full_reward=$gross_reward / ($total_machines * $total_days);
        $sub_reward=$after_gross / ($total_machines * $total_days);

        return [
            'reward_per_day'     => $full_reward,
            'sub_reward_per_day'     => $sub_reward,
        ];
    }
    private function referralCommissionPerUser($id,$user_id)
    {
        $config = MonthlyConfig::findOrFail($id);
        $status = $config->status;
        $configId = $id;

        $period_start = Carbon::parse($config->period_start);
        $period_end = Carbon::parse($config->period_end);

        $hashrate = $this->getHashrateForPeriod($period_start, $period_end);
        $total_machines = $this->getTotalMachinesDuringPeriod($period_end);

        $result = $this->calculateRewards($config, $period_start, $period_end);
        $reward_per_day=$result['reward_per_day'];
        $sub_reward_per_day=$result['sub_reward_per_day'];

    // Get the investments within the given period
        $investments = DB::table('users')
        ->select([
            'invest_histories.id',
            'users.id as user_id',
            'invest_histories.plan_id',
            'invest_histories.invest_amount',
            'invest_histories.quantity',
            'invest_histories.created_at',
            'users.email',
            DB::raw("CONCAT(users.firstname, ' ', users.lastname) AS user_name"),
            'investment_plans.plan_name'
        ])
        ->leftJoin('invest_histories', function($join) use ($period_end) {
            $join->on('users.id', '=', 'invest_histories.user_id')
            ->whereDate('invest_histories.created_at', '<=', $period_end)
            ->whereDate('invest_histories.plan_expiry_date', '>=', $period_end);
        })
        ->leftJoin('investment_plans', 'investment_plans.id', '=', 'invest_histories.plan_id')
        ->orderBy('users.id')
        ->get();

        //this is the calculation of the each user transaction
        $processedInvestments = $investments->map(function ($investment) use ($period_start, $period_end, $reward_per_day, $hashrate) {
            if ($investment->created_at) {
                $investmentDate = Carbon::parse($investment->created_at); 
                $purchased_date = $investmentDate->format('Y-m-d');

                $today = Carbon::today();
                $reward_cutoff = $today->lessThan($period_end) ? $today : $period_end;

                $proposed_start = $investmentDate->copy()->addDay();
                $effective_start = $proposed_start->greaterThan($period_start) ? $proposed_start : $period_start;

                $effective_end = $reward_cutoff;

                $effective_start = $effective_start->copy()->startOfDay();
                $effective_end = $effective_end->copy()->startOfDay();

                $days_invested = $effective_start->lte($effective_end)
                ? $effective_start->diffInDays($effective_end) + 1
                : 0;

                $calculated_reward = $reward_per_day * $days_invested * $investment->quantity;
            } else {
                $days_invested = 0;
                $purchased_date = '';
                $calculated_reward = '';
                $effective_start = '';
            }

            return [
                'id' => $investment->id ?? '',
                'user_id' => $investment->user_id,
                'planId' => $investment->plan_id ?? '',
                'user_name' => $investment->user_name,
                'invest_amount' => $investment->invest_amount ?? '',
                'quantity' => $investment->quantity ?? '',
                'days_invested' => $days_invested,
                'calculated_reward' => $calculated_reward !== '' ? number_format($calculated_reward, 4, '.', '') : '',
                'purchased_date' => $purchased_date,
                'effective_start_date' => $effective_start !== '' ? $effective_start->format('Y-m-d') : '',
                'effective_end_date' => $investment->created_at ? $period_end->format('Y-m-d') : '',
                'hashrate' => $investment->created_at ? $hashrate : '',
            ];
        });

        $referralCommissions = Referral::where('commission_type', 'profit_commission')
        ->whereIn('level', [1, 2, 3])
        ->pluck('commission', 'level');

        $users=User::where('id',$user_id)->pluck('id');

        $referralReport=array();
        foreach($users as $user){
            $checklvl = getDirectReferralUsers($user);
            $countofref=$checklvl->count();
            if($countofref == 0){
                continue;
            }
            $userData = User::find($user);
            $referralReport[$user]['agent'] = [
                'user_id' => $userData->id,
                'user_name' => $userData->firstname.' '.$userData->lastname,
                'user_email' => $userData->email,
                'purchased_date' => null,
                'start_date' => null,
                'end_date' => null,
                'quantity' => null,
                'commission' => null,
                'referral_name' => null,
            ];

            $result = $this->processLevel1Referrals($user, $processedInvestments, $referralCommissions);
            $referralReport[$user]['level1'] = $result['level1'];
        }
        return $referralReport;
    }
    private function processLevel1Referrals($userId, $userInvestments, $referralCommissions)
    {
        $level1Users = getDirectReferralUsers($userId);
        $level1Data = [];

        foreach ($level1Users as $level1User) {
            $rewardEntry = $userInvestments->firstWhere('user_id', $level1User->id);
            if (!$rewardEntry) continue;
            $reward = $rewardEntry ? (float)$rewardEntry['calculated_reward'] : 0;
            $commission = ($referralCommissions[1] ?? 0) / 100 * $reward;

            $level2Result = $this->processLevel2Referrals($level1User->id, $userInvestments, $referralCommissions);

            $level1Data[] = [
                'user_id' => $level1User->id,
                'user_email' => $level1User->email,
                'user_name' => $level1User->firstname . ' ' . $level1User->lastname,
                'reward' => $reward,
                'commission_rate' => $referralCommissions[1] ?? 0,
                'commission' => $commission,
                'purchased_date' => $rewardEntry['purchased_date'],
                'start_date' => $rewardEntry['effective_start_date'],
                'end_date' => $rewardEntry['effective_end_date'],
                'quantity' => $rewardEntry['quantity'],
                'level2' => $level2Result['level2']
            ];

        }

        return [
            'level1' => $level1Data,
        ];
    }

    private function processLevel2Referrals($level1UserId, $userInvestments, $referralCommissions)
    {
        $level2Users = getDirectReferralUsers($level1UserId);
        $level2Data = [];

        foreach ($level2Users as $level2User) {
            $rewardEntry = $userInvestments->firstWhere('user_id', $level2User->id);
            if (!$rewardEntry) continue;
            $reward = $rewardEntry ? (float)$rewardEntry['calculated_reward'] : 0;
            $commission = ($referralCommissions[2] ?? 0) / 100 * $reward;

            $level3Result = $this->processLevel3Referrals($level2User->id, $userInvestments, $referralCommissions);

            $level2Data[] = [
                'user_id' => $level2User->id,
                'user_email' => $level2User->email,
                'user_name' => $level2User->firstname . ' ' . $level2User->lastname,
                'reward' => $reward,
                'commission_rate' => $referralCommissions[2] ?? 0,
                'commission' => $commission,
                'purchased_date' => $rewardEntry['purchased_date'],
                'start_date' => $rewardEntry['effective_start_date'],
                'end_date' => $rewardEntry['effective_end_date'],
                'quantity' => $rewardEntry['quantity'],
                'level3' => $level3Result['level3']
            ];
        }

        return [
            'level2' => $level2Data,
        ];
    }

    private function processLevel3Referrals($level2UserId, $userInvestments, $referralCommissions)
    {
        $level3Users = getDirectReferralUsers($level2UserId);
        $level3Data = [];

        foreach ($level3Users as $level3User) {
            $rewardEntry = $userInvestments->firstWhere('user_id', $level3User->id);
            if (!$rewardEntry) continue;
            $reward = $rewardEntry ? (float)$rewardEntry['calculated_reward'] : 0;
            $commission = ($referralCommissions[3] ?? 0) / 100 * $reward;

            $level3Data[] = [
                'user_id' => $level3User->id,
                'user_email' => $level3User->email,
                'user_name' => $level3User->firstname . ' ' . $level3User->lastname,
                'reward' => $reward,
                'commission_rate' => $referralCommissions[3] ?? 0,
                'commission' => $commission,
                'purchased_date' => $rewardEntry['purchased_date'],
                'start_date' => $rewardEntry['effective_start_date'],
                'end_date' => $rewardEntry['effective_end_date'],
                'quantity' => $rewardEntry['quantity']
            ];

        }

        return [
            'level3' => $level3Data,
        ];
    }

    private function summaryPerUser($id,$user_id)
    {
        $config = MonthlyConfig::findOrFail($id);
        $status = $config->status;
        $configId = $id;

        $period_start = Carbon::parse($config->period_start);
        $period_end = Carbon::parse($config->period_end);

        $hashrate = $this->getHashrateForPeriod($period_start, $period_end);

        $result = $this->calculateRewards($config, $period_start, $period_end);
        $reward_per_day=$result['reward_per_day'];
        $sub_reward_per_day=$result['sub_reward_per_day'];

        $investments = DB::table('invest_histories')
        ->select([
            'invest_histories.id',
            'invest_histories.user_id',
            'invest_histories.plan_id',
            'invest_histories.invest_amount',
            'invest_histories.quantity',
            'invest_histories.created_at',
            DB::raw("CONCAT(users.firstname, ' ', users.lastname) AS user_name"),
            'investment_plans.plan_name'
        ])
        ->join('users', 'users.id', '=', 'invest_histories.user_id')
        ->join('investment_plans', 'investment_plans.id', '=', 'invest_histories.plan_id')
        ->whereDate('invest_histories.created_at', '<=', $period_end)
        ->whereDate('invest_histories.plan_expiry_date', '>=', $period_end)
        ->where('invest_histories.user_id', $user_id)
        ->orderBy('invest_histories.created_at', 'desc')
        ->get();

        $total_days_invested = 0;
        $total_reward = 0;
        $total_quantity=0;
        $processedInvestments = $investments->map(function ($investment) use ($period_start, $period_end, $sub_reward_per_day,$hashrate,&$total_days_invested, &$total_reward,&$total_quantity) {
            $investmentDate = Carbon::parse($investment->created_at); 
            $purchased_date = $investmentDate->format('Y-m-d');

            $today = Carbon::today();
            $reward_cutoff = $today->lessThan($period_end) ? $today : $period_end;

            $proposed_start = $investmentDate->copy()->addDay();
            $effective_start = $proposed_start->greaterThan($period_start) ? $proposed_start : $period_start;

            $effective_end = $reward_cutoff;

            $effective_start = $effective_start->copy()->startOfDay();
            $effective_end = $effective_end->copy()->startOfDay();

            $days_invested = $effective_start->lte($effective_end)
            ? $effective_start->diffInDays($effective_end) + 1
            : 0;

            $calculated_reward = $sub_reward_per_day * $days_invested * $investment->quantity;

            $total_days_invested += $days_invested;
            $total_reward += $calculated_reward;
            $total_quantity += $investment->quantity;

            return [
                'id' => $investment->id,
                'user_id' => $investment->user_id,
                'planId' => $investment->plan_id,
                'user_name' => $investment->user_name,
                'invest_amount' => $investment->invest_amount,
                'quantity' => $investment->quantity,
                'days_invested' => $days_invested,
                'calculated_reward' => number_format($calculated_reward, 4, '.', ''),
                'purchased_date' => $purchased_date,
                'effective_start_date' => $effective_start->format('Y-m-d'), 
                'effective_end_date' => $period_end->format('Y-m-d'), 
                'hashrate' => $hashrate, 
            ];
        });

        return $processedInvestments;
    }
}