<?php

namespace App\Http\Controllers\Admin;

use Illuminate\Http\Request;
use App\Models\GeneralConfig;
use App\Http\Controllers\Controller;
use Yajra\DataTables\DataTables;

class MonthlyGeneralController extends Controller
{
    public function index()
    {
        return view('admin.monthly_general.index');
    }

    public function list(Request $request)
    {
        $monthlyPlans = GeneralConfig::query();

        return DataTables::of($monthlyPlans)
        ->editColumn('period_start', function ($general) {
            return $general->period_start ? $general->period_start->format('Y-m-d') : '-';
        })
        ->editColumn('period_end', function ($general) {
            return $general->period_end ? $general->period_end->format('Y-m-d') : '-';
        })
        ->editColumn('total_hashrate', function ($general) {
            return $general->total_hashrate ?? '-';
        })
        ->editColumn('total_coin_generated', function ($general) {
            return $general->total_coin_generated ?? '-';
        })
        ->editColumn('created_at', function ($general) {
            return $general->created_at->format('Y-m-d H:i:s');
        })
        ->addColumn('action', function ($general) {
            $editBtn = '<a href="' . route('admin.monthly.general.edit', $general->id) . '" class="btn btn-sm btn-outline-primary" title="Edit">
            <i class="bi bi-pencil-square"></i>
            </a>';

            $deleteForm = '<form action="' . route('admin.monthly.general.destroy', $general->id) . '" method="POST" class="d-inline">
            ' . csrf_field() . '
            ' . method_field('DELETE') . '
            <button type="submit" class="btn btn-sm btn-outline-danger" onclick="return confirm(\'Are you sure you want to delete this entry?\')" title="Delete">
            <i class="bi bi-trash"></i>
            </button>
            </form>';

            return '<div class="btn-group" role="group">' . $editBtn . $deleteForm . '</div>';
        })
        ->rawColumns(['action'])
        ->make(true);
    }

    public function create()
    {
        return view('admin.monthly_general.create');
    }

    public function store(Request $request)
    {
        $request->validate([
            'period_start' => 'required|date',
            'period_end' => 'required|date|after_or_equal:period_start',
            'total_hashrate' => 'required',
            'total_coin_generated' => 'required',
            'total_machine' => 'required',
        ]);

        GeneralConfig::create([
            'period_start' => $request->period_start,
            'period_end' => $request->period_end,
            'total_hashrate' => $request->total_hashrate,
            'total_coin_generated' => $request->total_coin_generated,
            'total_machine' => $request->total_machine,
        ]);

        return redirect()->route('admin.monthly.general.index')->with('success', 'Monthly config created successfully.');
    }

    public function edit($id)
    {
        $config = GeneralConfig::findOrFail($id);
        return view('admin.monthly_general.edit', compact('config'));
    }

    public function update(Request $request, $id)
    {
        $request->validate([
            'period_start' => 'required|date',
            'period_end' => 'required|date|after_or_equal:period_start',
            'total_hashrate' => 'required',
            'total_coin_generated' => 'required',
            'total_machine' => 'required',
        ]);

        $config = GeneralConfig::findOrFail($id);
        $config->update([
            'period_start' => $request->period_start,
            'period_end' => $request->period_end,
            'total_hashrate' => $request->total_hashrate,
            'total_machine' => $request->total_machine,
            'total_coin_generated' => $request->total_coin_generated,
        ]);

        return redirect()->route('admin.monthly.general.index')->with('success', 'Monthly config updated successfully.');
    }

    public function destroy($id)
    {
        $config = GeneralConfig::findOrFail($id);
        $config->delete();

        return redirect()->route('admin.monthly.general.index')->with('success', 'Monthly config deleted successfully.');
    }
}
