<?php

namespace App\Http\Controllers\Admin;

use App\Http\Controllers\Controller;
use App\Models\Deposit;
use App\Models\InvestHistory;
use App\Models\InvestmentPlan;
use App\Models\Language;
use App\Models\Order;
use App\Models\ReferralBonus;
use App\Models\SupportTicket;
use App\Models\Transaction;
use App\Models\User;
use App\Models\UserKyc;
use App\Traits\Notify;
use Carbon\Carbon;
use Illuminate\Http\Request;
use App\Traits\Upload;
use Illuminate\Support\Facades\Auth;
use Illuminate\Support\Facades\DB;


class DashboardController extends Controller
{
    use Upload, Notify;

    public function index()
    {
        $data['firebaseNotify'] = config('firebase');
        $data['latestUser'] = User::where('total_invest','>',1)->orderBy('total_invest','desc')->limit(5)->get();
        $statistics['schedule'] = $this->dayList();
        $data['bestSellingProducts'] = DB::table('order_items')
        ->select('products.id','products.thumbnail_image','products.slug','products.driver','products.price','categories.name as category_name', 'product_details.title', DB::raw('SUM(order_items.quantity) as total_sold'))
        ->join('products', 'order_items.product_id', '=', 'products.id')
        ->join('categories', 'products.category_id', '=', 'categories.id')
        ->join('product_details', 'products.id', '=', 'product_details.product_id')
        ->groupBy('products.id', 'product_details.title')
        ->orderByDesc('total_sold')
        ->take(5)
        ->get();
        $data['topPlans'] = InvestmentPlan::with('investment')
        ->withSum('investment','invest_amount')
        ->orderByDesc('investment_sum_invest_amount')
        ->limit(5)
        ->get();
        $data['language'] = Language::where('default_status',1)->first();
        return view('admin.dashboard-alternative', $data, compact("statistics"));
    }

    public function topInvestors(Request $request)
    {
        $start = $request->input('start_date');
        $end = $request->input('end_date');

        $latestUser = User::where('total_invest', '>', 1)
        ->when($start && $end, function ($query) use ($start, $end) {
            $query->whereBetween('created_at', [$start, $end]);
        })
        ->orderByDesc('total_invest')
        ->limit(5)
        ->get();

        return response()->json(['users' => $latestUser]);
    }


    public function monthlyDepositWithdraw(Request $request)
    {
        $keyDataset = $request->keyDataset;

        $dailyDeposit = $this->dayList();

        Deposit::when($keyDataset == '0', function ($query) {
            $query->whereMonth('created_at', Carbon::now()->month);
        })
        ->when($keyDataset == '1', function ($query) {
            $lastMonth = Carbon::now()->subMonth();
            $query->whereMonth('created_at', $lastMonth->month);
        })
        ->select(
            DB::raw('SUM(payable_amount_in_base_currency) as totalDeposit'),
            DB::raw('DATE_FORMAT(created_at,"Day %d") as date')
        )
        ->groupBy(DB::raw("DATE(created_at)"))
        ->get()->map(function ($item) use ($dailyDeposit) {
            $dailyDeposit->put($item['date'], $item['totalDeposit']);
        });

        return response()->json([
            "totalDeposit" => currencyPosition($dailyDeposit->sum()),
            "dailyDeposit" => $dailyDeposit,
        ]);
    }

    public function saveToken(Request $request)
    {
        $admin = Auth::guard('admin')->user()
        ->fireBaseToken()
        ->create([
            'token' => $request->token,
        ]);
        return response()->json([
            'msg' => 'token saved successfully.',
        ]);
    }


    public function dayList()
    {
        $totalDays = Carbon::now()->endOfMonth()->format('d');
        $daysByMonth = [];
        for ($i = 1; $i <= $totalDays; $i++) {
            array_push($daysByMonth, ['Day ' . sprintf("%02d", $i) => 0]);
        }

        return collect($daysByMonth)->collapse();
    }

    protected function followupGrap($todaysRecords, $lastDayRecords = 0)
    {

        if (0 < $lastDayRecords) {
            $percentageIncrease = (($todaysRecords - $lastDayRecords) / $lastDayRecords) * 100;
        } else {
            $percentageIncrease = 0;
        }
        if ($percentageIncrease > 0) {
            $class = "bg-soft-success text-success";
        } elseif ($percentageIncrease < 0) {
            $class = "bg-soft-danger text-danger";
        } else {
            $class =  "bg-soft-secondary text-body";
        }

        return [
            'class' => $class,
            'percentage' => round($percentageIncrease, 2)
        ];
    }




    public function chartUserRecords()
    {
        $userRecord = collect(User::selectRaw('COUNT(id) AS totalUsers')
            ->selectRaw('COUNT(CASE WHEN DATE(created_at) = CURDATE() THEN id END) AS currentDateUserCount')
            ->selectRaw('COUNT(CASE WHEN DATE(created_at) = DATE(DATE_SUB(NOW(), INTERVAL 1 DAY)) THEN id END) AS previousDateUserCount')
            ->get()
            ->makeHidden(['last-seen-activity', 'fullname'])
            ->toArray())->collapse();

        $followupGrap = $this->followupGrap($userRecord['currentDateUserCount'], $userRecord['previousDateUserCount']);

        $userRecord->put('followupGrapClass', $followupGrap['class']);
        $userRecord->put('followupGrap', $followupGrap['percentage']);
        $userRecord['chartPercentageIncDec'] = fractionNumber($userRecord['totalUsers'] - $userRecord['currentDateUserCount'], false);

        $all_time_data = DB::table('users')
        ->select(DB::raw('DATE_FORMAT(created_at,"%e %b") as date'), DB::raw('count(*) as count'))
        ->orderBy('created_at', 'asc')
        ->groupBy('date')
        ->get();

        $data_dates = $all_time_data->pluck('date');
        $data_counts = $all_time_data->pluck('count');

        return response()->json([
            'userRecord' => $userRecord,
        'current_month_data_dates' => $data_dates, // keep the variable names consistent with JS
        'current_month_datas' => $data_counts
    ]);
    }


    public function chartTicketRecords()
    {
        $currentMonth = Carbon::now()->format('Y-m');
        $ticketRecord = collect(SupportTicket::selectRaw('COUNT(id) AS totalTickets')
            ->selectRaw('COUNT(CASE WHEN DATE(created_at) = CURDATE() THEN id END) AS currentDateTicketsCount')
            ->selectRaw('COUNT(CASE WHEN DATE(created_at) = DATE(DATE_SUB(NOW(), INTERVAL 1 DAY)) THEN id END) AS previousDateTicketsCount')
            ->selectRaw('count(CASE WHEN status = 2  THEN status END) AS replied')
            ->selectRaw('count(CASE WHEN status = 1  THEN status END) AS answered')
            ->selectRaw('count(CASE WHEN status = 0  THEN status END) AS pending')
            ->get()
            ->toArray())->collapse();

        $followupGrap = $this->followupGrap($ticketRecord['currentDateTicketsCount'], $ticketRecord['previousDateTicketsCount']);
        $ticketRecord->put('followupGrapClass', $followupGrap['class']);
        $ticketRecord->put('followupGrap', $followupGrap['percentage']);

        $current_month_data = DB::table('support_tickets')
        ->select(DB::raw('DATE_FORMAT(created_at,"%e %b") as date'), DB::raw('count(*) as count'))
        ->where(DB::raw('DATE_FORMAT(created_at, "%Y-%m")'), $currentMonth)
        ->orderBy('created_at', 'asc')
        ->groupBy('date')
        ->get();

        $current_month_data_dates = $current_month_data->pluck('date');
        $current_month_datas = $current_month_data->pluck('count');
        $ticketRecord['chartPercentageIncDec'] = fractionNumber($ticketRecord['totalTickets'] - $ticketRecord['currentDateTicketsCount'], false);
        return response()->json(['ticketRecord' => $ticketRecord, 'current_month_data_dates' => $current_month_data_dates, 'current_month_datas' => $current_month_datas]);
    }
    public function chartPlansPurchased(Request $request)
    {
        $start = $request->input('start_date');
        $end = $request->input('end_date');

        $query = DB::table('invest_histories')
        ->join('investment_plans', 'invest_histories.plan_id', '=', 'investment_plans.id')
        ->select('investment_plans.plan_name', DB::raw('SUM(invest_histories.quantity) as total_quantity'))
        ->where('invest_histories.status', 1);

        if ($start && $end) {
            $startDate = Carbon::parse($start)->startOfDay();
        $endDate = Carbon::parse($end)->endOfDay(); // 🟢 This ensures today is fully included
        $query->whereBetween('invest_histories.created_at', [$startDate, $endDate]);
    }

    $plans = $query->groupBy('investment_plans.plan_name')
    ->orderByDesc('total_quantity')
    ->get();

    return response()->json([
        'labels' => $plans->pluck('plan_name'),
        'data' => $plans->pluck('total_quantity'),
    ]);
}



public function chartKycRecords()
{
    $currentMonth = Carbon::now()->format('Y-m');
    $kycRecords = collect(UserKyc::selectRaw('COUNT(id) AS totalKYC')
        ->selectRaw('COUNT(CASE WHEN DATE(created_at) = CURDATE() THEN id END) AS currentDateKYCCount')
        ->selectRaw('COUNT(CASE WHEN DATE(created_at) = DATE(DATE_SUB(NOW(), INTERVAL 1 DAY)) THEN id END) AS previousDateKYCCount')
        ->selectRaw('count(CASE WHEN status = 0  THEN status END) AS pendingKYC')
        ->get()
        ->toArray())->collapse();
    $followupGrap = $this->followupGrap($kycRecords['currentDateKYCCount'], $kycRecords['previousDateKYCCount']);
    $kycRecords->put('followupGrapClass', $followupGrap['class']);
    $kycRecords->put('followupGrap', $followupGrap['percentage']);


    $current_month_data = DB::table('user_kycs')
    ->select(DB::raw('DATE_FORMAT(created_at,"%e %b") as date'), DB::raw('count(*) as count'))
    ->where(DB::raw('DATE_FORMAT(created_at, "%Y-%m")'), $currentMonth)
    ->orderBy('created_at', 'asc')
    ->groupBy('date')
    ->get();

    $current_month_data_dates = $current_month_data->pluck('date');
    $current_month_datas = $current_month_data->pluck('count');
    $kycRecords['chartPercentageIncDec'] = fractionNumber($kycRecords['totalKYC'] - $kycRecords['currentDateKYCCount'], false);
    return response()->json(['kycRecord' => $kycRecords, 'current_month_data_dates' => $current_month_data_dates, 'current_month_datas' => $current_month_datas]);
}

public function chartTransactionRecords()
{
    $currentMonth = Carbon::now()->format('Y-m');

    $transaction = collect(Transaction::selectRaw('COUNT(id) AS totalTransaction')
        ->selectRaw('COUNT(CASE WHEN DATE(created_at) = CURDATE() THEN id END) AS currentDateTransactionCount')
        ->selectRaw('COUNT(CASE WHEN DATE(created_at) = DATE(DATE_SUB(NOW(), INTERVAL 1 DAY)) THEN id END) AS previousDateTransactionCount')
        ->whereRaw('YEAR(created_at) = YEAR(NOW()) AND MONTH(created_at) = MONTH(NOW())')
        ->get()
        ->toArray())
    ->collapse();

    $followupGrap = $this->followupGrap($transaction['currentDateTransactionCount'], $transaction['previousDateTransactionCount']);
    $transaction->put('followupGrapClass', $followupGrap['class']);
    $transaction->put('followupGrap', $followupGrap['percentage']);


    $current_month_data = DB::table('transactions')
    ->select(DB::raw('DATE_FORMAT(created_at,"%e %b") as date'), DB::raw('count(*) as count'))
    ->where(DB::raw('DATE_FORMAT(created_at, "%Y-%m")'), $currentMonth)
    ->orderBy('created_at', 'asc')
    ->groupBy('date')
    ->get();
    $current_month_data_dates = $current_month_data->pluck('date');
    $current_month_datas = $current_month_data->pluck('count');
    $transaction['chartPercentageIncDec'] = fractionNumber($transaction['totalTransaction'] - $transaction['currentDateTransactionCount'], false);
    return response()->json(['transactionRecord' => $transaction, 'current_month_data_dates' => $current_month_data_dates, 'current_month_datas' => $current_month_datas]);
}

public function chartPurchasePeople()
{
    $totalUsers = Transaction::distinct('user_id')->count('user_id');

    $data = [
        'total' => $totalUsers,
        'followupGrapClass' => 'bg-soft-success text-success',
        'followupGrap' => '100',
        'chartPercentageIncDec' => 'All Time'
    ];

    return response()->json(['purchasePeopleRecord' => $data, 'current_month_data_dates' => [], 'current_month_datas' => []]);
}
public function chartPurchaseUnits()
{
    $totalUnits = DB::table('invest_histories')
        ->where('status', 1) // Only active investments
        ->where(function ($query) {
            $query->where('plan_period_is_lifetime', 1) // Lifetime plans are always valid
                  ->orWhereDate('plan_expiry_date', '>=', now()); // Or still active (not expired)
              })
        ->sum('quantity');

        $data = [
            'total' => intval($totalUnits),
            'followupGrapClass' => 'bg-soft-success text-success',
            'followupGrap' => '100',
            'chartPercentageIncDec' => 'All Time'
        ];

        return response()->json(['purchaseUnitRecord' => $data, 'current_month_data_dates' => [], 'current_month_datas' => []]);
    }


    public function chartLoginHistory()
    {
        $userLoginsData = DB::table('user_logins')
        ->whereDate('created_at', '>=', now()->subDays(30))
        ->select('browser', 'os', 'get_device')
        ->get();

        $userLoginsBrowserData = $userLoginsData->groupBy('browser')->map->count();
        $data['browserKeys'] = $userLoginsBrowserData->keys();
        $data['browserValue'] = $userLoginsBrowserData->values();

        $userLoginsOSData = $userLoginsData->groupBy('os')->map->count();
        $data['osKeys'] = $userLoginsOSData->keys();
        $data['osValue'] = $userLoginsOSData->values();

        $userLoginsDeviceData = $userLoginsData->groupBy('get_device')->map->count();
        $data['deviceKeys'] = $userLoginsDeviceData->keys();
        $data['deviceValue'] = $userLoginsDeviceData->values();

        return response()->json(['loginPerformance' => $data]);
    }

    public function investHistory(Request $request)
    {
        $startDate = \Illuminate\Support\Carbon::parse($request->start_date);
        $endDate = Carbon::parse($request->end_date)->endOfDay();

        $diffInDays = $startDate->diffInDays($endDate);
        $groupBy = $diffInDays <= 1 ? 'HOUR' : 'DATE';
        $dateFormat = $diffInDays <= 1 ? 'ga' : 'd-M';


        $invests = InvestHistory::whereBetween('created_at', [$startDate, $endDate])
        ->select(
            DB::raw($groupBy . '(created_at) as period'),
            DB::raw('SUM(invest_amount) as total'),
        )
        ->groupBy('period')
        ->get();
        $labels = [];
        $investData = [];

        $total =0;
        foreach ($invests as $data) {
            $period = $data->period;
            $formattedPeriod = $diffInDays <= 1 ? date($dateFormat, strtotime($period . ':00')) : date($dateFormat, strtotime($period));
            $labels[] = $formattedPeriod;
            $investData[] = $data->total;
            $total += $data->total;
        }

        $totalInvest = InvestHistory::select(DB::raw('SUM(invest_amount) as totalInvest'))->value('totalInvest');


        return response()->json([
            'planInvest' => $investData,
            'schedule' => $labels,
            'thisMonthInvest' => currencyPosition(getAmount($total)),
            'totalInvest' => currencyPosition($totalInvest)
        ]);
    }
    public function hourList()
    {
        $hoursByDay = [];
        for ($hour = 0; $hour < 24; $hour++) {
            $formattedHour = sprintf('%02d:00 %s', $hour, $hour < 12 ? 'AM' : 'PM');
            $hoursByDay[$formattedHour] = '00';
        }

        return $hoursByDay;
    }

    public function referralBonusHistory()
    {
        $hourly = $this->hourList();
        $labels = $this->hourList();
        $todayBonus = ReferralBonus::whereDate('created_at', Carbon::today())
        ->select(
            DB::raw('SUM(amount) as perHourBonus'),
            DB::raw('DATE_FORMAT(created_at, "%H:00 %p") as hourOfDay')
        )
        ->groupBy(DB::raw("HOUR(created_at)"))
        ->get();

        foreach ($todayBonus as $item){
            $hourly[$item['hourOfDay']] =  $item['perHourBonus'];
        }

        $perHourYesterdayBonus = $this->hourList();
        $yesterdayBonus = ReferralBonus::whereDate('created_at', Carbon::yesterday())
        ->select(
            DB::raw('SUM(amount) as perHourBonus'),
            DB::raw('DATE_FORMAT(created_at, "%H:00 %p") as hourOfDay')
        )
        ->groupBy(DB::raw("HOUR(created_at)"))
        ->get();

        foreach ($yesterdayBonus as $item){
            $perHourYesterdayBonus[$item['hourOfDay']] =  $item['perHourBonus'];
        }
        $totalBonus = ReferralBonus::select(
            DB::raw('SUM(amount) as totalAmount'),
        )->first();
        return response()->json([
            'perHourBonus' => array_values($hourly),
            'labels' => array_keys($labels),
            'perHourYesterdayBonus' => array_values($perHourYesterdayBonus),
            'totalBonus' => currencyPosition($totalBonus['totalAmount']),
            'todayBonus' => currencyPosition($todayBonus->sum('perHourBonus'))
        ]);

    }


    public function getSalesRevenueHistory(Request $request)
    {
        $startDate = Carbon::parse($request->start_date);
        $endDate = Carbon::parse($request->end_date)->endOfDay();

        // grouping type based on date range
        $diffInDays = $startDate->diffInDays($endDate);
        $previousStartDate = $startDate->copy()->subDays($diffInDays + 1);
        $previousEndDate = $endDate->copy()->subDays($diffInDays + 1);
        $groupBy = $diffInDays <= 1 ? 'HOUR' : 'DATE';
        $dateFormat = $diffInDays <= 1 ? 'ga' : 'd-M';

        $salesData = Order::where('payment_status', 1)
        ->whereBetween('created_at', [$startDate, $endDate])
        ->where('payment_status', 1)
        ->selectRaw($groupBy . '(created_at) as period,
           COUNT(*) as item_count,
           SUM(total) as total')
        ->groupBy('period')
        ->get();


        $labels = [];
        $revenueData = [];
        $ordersData = [];

        foreach ($salesData as $data) {
            $period = $data->period;
            $formattedPeriod = $diffInDays <= 1 ? date($dateFormat, strtotime($period . ':00')) : date($dateFormat, strtotime($period));
            $labels[] = $formattedPeriod;
            $revenueData[] = $data->total;
            $ordersData[] = $data->item_count;
        }

        $currentPeriodSalesData = Order::where('payment_status', 1)
        ->whereBetween('created_at', [$startDate, $endDate])
        ->where('payment_status', 1)
        ->whereBetween('created_at', [$startDate, $endDate])
        ->selectRaw('SUM(total) as total,
           COUNT(*) as item_count')
        ->first();

        $previousPeriodSalesData = Order::where('payment_status', 1)
        ->whereBetween('created_at', [$startDate, $endDate])
        ->where('payment_status', 1)
        ->whereBetween('created_at', [$previousStartDate, $previousEndDate])
        ->selectRaw('SUM(total) as total,
           COUNT(*) as item_count')
        ->first();

        $revenuePercentageChange = '0%';
        $revenuePercentageValue = $currentPeriodSalesData->total - $previousPeriodSalesData->total;
        $orderPercentageChange = '0%';
        $orderPercentageValue = $currentPeriodSalesData->item_count - $previousPeriodSalesData->item_count;

        if ($previousPeriodSalesData->total > 0) {
            $revenuePercentageChangeValue = round((($currentPeriodSalesData->total - $previousPeriodSalesData->total) / $previousPeriodSalesData->total) * 100, 2);
            $revenuePercentageChange = $revenuePercentageChangeValue . '%';
        }
        if ($previousPeriodSalesData->item_count > 0) {
            $orderPercentageChangeValue = round((($currentPeriodSalesData->item_count - $previousPeriodSalesData->item_count) / $previousPeriodSalesData->item_count) * 100, 2);
            $orderPercentageChange = $orderPercentageChangeValue . '%';
        }

        $response = [
            'labels' => $labels,
            'datasets' => [
                [
                    'label' => 'Revenue',
                    'data' => $revenueData,
                    'backgroundColor' => '#377dff',
                    'hoverBackgroundColor' => '#377dff',
                    'borderColor' => '#377dff',
                    'maxBarThickness' => 10,
                ],
                [
                    'label' => 'Orders',
                    'data' => $ordersData,
                    'backgroundColor' => '#e7eaf3',
                    'borderColor' => '#e7eaf3',
                    'maxBarThickness' => 10,
                ],
            ],
            'revenuePercentageChange' => $revenuePercentageChange,
            'revenuePercentageValue' => $revenuePercentageValue,
            'orderPercentageChange' => $orderPercentageChange,
            'orderPercentageValue' => $orderPercentageValue,
        ];

        return response()->json($response);
    }

    public function chartBrowserHistory(Request $request)
    {
        $startDate = $request->startDate;
        $endDate = $request->endDate;

        $userLoginsData = DB::table('user_logins')
        ->whereBetween('created_at', [$startDate, $endDate])
        ->select('browser', 'os', 'get_device')
        ->get();

        $userLoginsBrowserData = $userLoginsData->groupBy('browser')->map->count();
        $data['browserKeys'] = $userLoginsBrowserData->keys();
        $data['browserValue'] = $userLoginsBrowserData->values();

        return response()->json(['browserPerformance' => $data]);
    }

    public function chartOsHistory(Request $request)
    {
        $startDate = $request->startDate;
        $endDate = $request->endDate;

        $userLoginsData = DB::table('user_logins')
        ->whereBetween('created_at', [$startDate, $endDate])
        ->select('browser', 'os', 'get_device')
        ->get();

        $userLoginsOSData = $userLoginsData->groupBy('os')->map->count();
        $data['osKeys'] = $userLoginsOSData->keys();
        $data['osValue'] = $userLoginsOSData->values();

        return response()->json(['osPerformance' => $data]);
    }

    public function chartDeviceHistory(Request $request)
    {
        $startDate = $request->startDate;
        $endDate = $request->endDate;

        $userLoginsData = DB::table('user_logins')

        ->whereBetween('created_at', [$startDate, $endDate])
        ->select('browser', 'os', 'get_device')
        ->get();

        $userLoginsDeviceData = $userLoginsData->groupBy('get_device')->map->count();
        $data['deviceKeys'] = $userLoginsDeviceData->keys();
        $data['deviceValue'] = $userLoginsDeviceData->values();

        return response()->json(['deviceHistory' => $data]);
    }



}
